// controllers/payment.controller.js
const Payment = require("../models/Payment");
const axios = require("axios");
const crypto = require("crypto");
const moment = require('moment');
const { creditWallet } = require("../services/walletService");
const Wallet = require('../models/Wallet');
const WalletTransaction = require('../models/WalletTransaction');
// Environment variables
const UPI_GATEWAY_API_KEY = process.env.UPI_GATEWAY_API_KEY;
const UPI_GATEWAY_SECRET_KEY = process.env.UPI_GATEWAY_SECRET_KEY;
const UPI_CREATE_ORDER_URL = process.env.UPI_CREATE_ORDER_URL;
const UPI_CHECK_STATUS_URL = process.env.UPI_CHECK_STATUS_URL;
const BASE_URL = process.env.BASE_URL;
const FRONTEND_URL = process.env.FRONTEND_URL;

// ============================================
// HELPER FUNCTIONS
// ============================================

// Generate unique transaction ID
const generateTxnId = () => {
  return `TXN${Date.now()}${Math.floor(Math.random() * 1000)}`;
};

// Validate amount
const validateAmount = (amount) => {
  const amt = parseFloat(amount);
  return amt > 0 && amt <= 100000; // Max 1 lakh
};

// Validate mobile number (Indian)
const validateMobile = (mobile) => {
  const mobileRegex = /^[6-9]\d{9}$/;
  return mobileRegex.test(mobile);
};

// Validate email
const validateEmail = (email) => {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email);
};

// Verify webhook signature (for security)
const verifyWebhookSignature = (payload, signature) => {
  if (!UPI_GATEWAY_SECRET_KEY) return true; // Skip if no secret configured
  
  const hash = crypto
    .createHmac("sha256", UPI_GATEWAY_SECRET_KEY)
    .update(JSON.stringify(payload))
    .digest("hex");
  return hash === signature;
};

// ============================================
// CREATE PAYMENT ORDER
// ============================================
exports.createPaymentOrder = async (req, res) => {
  try {
    const {
      amount,
      productInfo,
      customerName,
      customerEmail,
      customerMobile,
      redirectUrl,
      udf1,
      udf2,
      udf3,
    } = req.body;

    // 🔐 USER FROM JWT (TRUSTED)
    const userId = req.auth.id;
    console.log("AUTH USER ID:", userId);

    // ================= VALIDATION =================
    if (!amount || !productInfo || !customerName || !customerEmail || !customerMobile) {
      return res.status(400).json({
        success: false,
        message: "Missing required fields",
      });
    }

    if (!validateAmount(amount)) {
      return res.status(400).json({
        success: false,
        message: "Invalid amount",
      });
    }

    if (!validateMobile(customerMobile)) {
      return res.status(400).json({
        success: false,
        message: "Invalid mobile number",
      });
    }

    if (!validateEmail(customerEmail)) {
      return res.status(400).json({
        success: false,
        message: "Invalid email",
      });
    }

    // ================= CREATE TXN ID =================
    const clientTxnId = generateTxnId();

    // ================= UPI PAYLOAD =================
    const upiPayload = {
      key: UPI_GATEWAY_API_KEY,
      client_txn_id: clientTxnId,
      amount: amount.toString(),
      p_info: productInfo,
      customer_name: customerName,
      customer_email: customerEmail,
      customer_mobile: customerMobile,
      redirect_url:
        redirectUrl || `${FRONTEND_URL}/payment/status/${clientTxnId}`,
      udf1: udf1 || "",
      udf2: udf2 || "",
      udf3: udf3 || "",
    };

    console.log("Creating UPI order:", {
      ...upiPayload,
      key: "***HIDDEN***",
    });

    // ================= CALL UPI GATEWAY =================
    const upiResponse = await axios.post(
      UPI_CREATE_ORDER_URL,
      upiPayload,
      { timeout: 15000 }
    );

    console.log("UPI Gateway Response:", upiResponse.data);

    if (!upiResponse.data?.status) {
      return res.status(400).json({
        success: false,
        message: upiResponse.data?.msg || "UPI order creation failed",
      });
    }

    // ================= SAVE PAYMENT =================
    const payment = new Payment({
      userId, // ✅ ObjectId from JWT
      clientTxnId,
      amount,
      productInfo,
      customerName,
      customerEmail,
      customerMobile,
      redirectUrl:
        redirectUrl || `${FRONTEND_URL}/payment/status/${clientTxnId}`,
      
      // ✅ FIX: Gateway response data properly mapped
      gatewayOrderId: upiResponse.data.data?.order_id,
      paymentUrl: upiResponse.data.data?.payment_url,
      upiIdHash: upiResponse.data.data?.upi_id_hash, // ✅ Correct path
      
      upiIntent: {
        bhimLink: upiResponse.data.data?.upi_intent?.bhim_link || "",
        phonepeLink: upiResponse.data.data?.upi_intent?.phonepe_link || "",
        paytmLink: upiResponse.data.data?.upi_intent?.paytm_link || "",
        gpayLink: upiResponse.data.data?.upi_intent?.gpay_link || "", // ✅ Added gpay
      },
      
      // ✅ IMPORTANT: Add txnDate
      txnDate: new Date(),
      
      status: "PENDING",
      paymentStatus: "initiated", // ✅ Added for clarity
      
      udf1,
      udf2,
      udf3,
      gatewayResponse: upiResponse.data,
    });

    await payment.save();

    console.log("✅ Payment saved:", payment._id);

    // ================= RESPONSE =================
    return res.status(201).json({
      success: true,
      message: "Payment order created successfully",
      data: {
        paymentId: payment._id,
        clientTxnId,
        amount,
        paymentUrl: upiResponse.data.data?.payment_url,
        upiIdHash: upiResponse.data.data?.upi_id_hash, // ✅ Return this for frontend
        upiIntent: upiResponse.data.data?.upi_intent,
        redirectUrl: payment.redirectUrl,
      },
    });

  } catch (error) {
    console.error("❌ Create Payment Error:", error.response?.data || error.message);

    if (error.code === "ECONNABORTED") {
      return res.status(504).json({
        success: false,
        message: "Payment gateway timeout",
      });
    }

    return res.status(500).json({
      success: false,
      message: "Failed to create payment order",
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
};


// ============================================
// CHECK PAYMENT STATUS
// ============================================


// Status mapping helper
const normalizeStatus = (gatewayStatus = "") => {
  const s = gatewayStatus.toString().toLowerCase();

  if (["success", "completed", "paid", "txn_success"].includes(s)) {
    return "SUCCESS";
  }

  if (["failed", "failure", "txn_failed"].includes(s)) {
    return "FAILED";
  }

  if (["expired", "timeout"].includes(s)) {
    return "EXPIRED";
  }

  return "PENDING";
};


// ========================================
// Check Payment Status - FIXED
// ========================================
exports.checkPaymentStatus = async (req, res) => {
  try {
    const { clientTxnId } = req.params;

    if (!clientTxnId) {
      return res.status(400).json({
        success: false,
        message: "Transaction ID is required",
      });
    }

    console.log("Checking payment status for:", clientTxnId);

    // Find payment in database
    const payment = await Payment.findOne({ clientTxnId });

    if (!payment) {
      return res.status(404).json({
        success: false,
        message: "Payment not found",
      });
    }

    // If already completed, return from database
    if (payment.status === "SUCCESS" || payment.status === "FAILED") {
      return res.json({
        success: true,
        data: {
          clientTxnId: payment.clientTxnId,
          status: payment.status,
          amount: payment.amount,
          utr: payment.utr || "",
          customerName: payment.customerName,
          createdAt: moment(payment.createdAt).format('DD-MM-YYYY HH:mm:ss'),
          updatedAt: moment(payment.updatedAt).format('DD-MM-YYYY HH:mm:ss'),
        },
      });
    }

    // Format txn_date
    const txnDate = moment().format('DD-MM-YYYY');

    console.log("Checking with gateway - txn_date:", txnDate);

    // Check with gateway
    const statusResponse = await axios.post(
      process.env.UPI_CHECK_STATUS_URL,
      {
        key: process.env.UPI_GATEWAY_API_KEY,
        client_txn_id: clientTxnId,
        txn_date: txnDate
      },
      {
        headers: {
          "Content-Type": "application/json",
        },
        timeout: 10000,
      }
    );

    console.log("Status check response:", statusResponse.data);

    // If gateway returns error
    if (!statusResponse.data.status) {
      console.error("Gateway status check failed:", statusResponse.data.msg);
      
      return res.json({
        success: true,
        data: {
          clientTxnId: payment.clientTxnId,
          status: payment.status,
          amount: payment.amount,
          utr: payment.utr || "",
          customerName: payment.customerName,
          message: statusResponse.data.msg,
        },
      });
    }

    // Normalize gateway status
    const gatewayStatus = statusResponse.data.data?.status || "created";
    const normalizedStatus = normalizeStatus(gatewayStatus);

    console.log(`Status update: ${gatewayStatus} → ${normalizedStatus}`);

    // Update payment in database
    payment.status = normalizedStatus;
    payment.utr = statusResponse.data.data?.upi_txn_id || payment.utr || "";
    payment.gatewayResponse = statusResponse.data;
    
    if (normalizedStatus === 'SUCCESS' && !payment.paidAt) {
      payment.paidAt = new Date();
    }
    
    await payment.save();

  

    return res.json({
      success: true,
      data: {
        clientTxnId: payment.clientTxnId,
        status: payment.status,
        amount: payment.amount,
        utr: payment.utr || "",
        customerName: payment.customerName,
        createdAt: moment(payment.createdAt).format('DD-MM-YYYY HH:mm:ss'),
        updatedAt: moment(payment.updatedAt).format('DD-MM-YYYY HH:mm:ss'),
      },
    });
  } catch (error) {
    console.error("Check Status Error:", error.message);

    return res.status(500).json({
      success: false,
      message: "Failed to check payment status",
      error: process.env.NODE_ENV === "development" ? error.message : "Internal server error",
    });
  }
};


// ============================================
// WEBHOOK HANDLER (Called by UPI Gateway)
// ============================================
exports.handleWebhook = async (req, res) => {
  try {
    console.log("========== WEBHOOK RECEIVED ==========");
    console.log("Headers:", req.headers);
    console.log("Body:", req.body);
    console.log("====================================");

    const { client_txn_id, status, txn_id } = req.body;
    const signature = req.headers["x-signature"] || req.body.signature;

    // 🔐 Verify signature (if enabled)
    if (process.env.UPI_GATEWAY_SECRET_KEY && signature) {
      const isValid = verifyWebhookSignature(req.body, signature);
      if (!isValid) {
        console.error("❌ Invalid webhook signature");
        return res.status(401).json({
          success: false,
          message: "Invalid signature"
        });
      }
      console.log("✅ Webhook signature verified");
    }

    if (!client_txn_id) {
      return res.status(400).json({
        success: false,
        message: "Missing client_txn_id"
      });
    }

    // Normalize status
    const normalizedStatus = status ? status.toUpperCase() : null;

    // Find payment
    const payment = await Payment.findOne({ clientTxnId: client_txn_id });

    if (!payment) {
      console.error("❌ Payment not found:", client_txn_id);
      return res.status(404).json({
        success: false,
        message: "Payment not found"
      });
    }

    /**
     * ✅ IDPOTENCY SAFE CHECK
     * If already SUCCESS/FAILED AND wallet already updated → exit
     */
    if (
      (payment.status === "SUCCESS" || payment.status === "FAILED") &&
      payment.walletUpdated
    ) {
      console.log("⚠️ Payment already fully processed:", payment.clientTxnId);
      return res.status(200).json({
        success: true,
        message: "Already processed"
      });
    }

    const oldStatus = payment.status;

    // Update payment
    if (normalizedStatus) {
      payment.status = normalizedStatus;
    }

    payment.webhookData = req.body;

    if (payment.status === "SUCCESS") {
      payment.paidAt = payment.paidAt || new Date();
      payment.gatewayTxnId = txn_id || payment.gatewayTxnId;
    }

    await payment.save();

    console.log(`✅ Payment updated: ${oldStatus} → ${payment.status}`);

    // 💰 CREDIT WALLET (ONLY HERE)
  if (payment.status === "SUCCESS" && !payment.walletUpdated) {
  await creditWallet({ payment });
}


    return res.status(200).json({
      success: true,
      message: "Webhook processed successfully"
    });
  } catch (error) {
    console.error("❌ Webhook Error:", error);
    return res.status(500).json({
      success: false,
      message: "Webhook processing failed",
      error: error.message
    });
  }
};


// ============================================
// GET PAYMENT DETAILS
// ============================================
exports.getPaymentDetails = async (req, res) => {
  try {
    const { clientTxnId } = req.params;

    if (!clientTxnId) {
      return res.status(400).json({
        success: false,
        message: "Transaction ID is required",
      });
    }

    const payment = await Payment.findOne({ clientTxnId }).populate("userId", "name email");

    if (!payment) {
      return res.status(404).json({
        success: false,
        message: "Payment not found",
      });
    }

    return res.status(200).json({
      success: true,
      data: payment,
    });
  } catch (error) {
    console.error("Get Payment Error:", error.message);
    return res.status(500).json({
      success: false,
      message: "Failed to get payment details",
      error: error.message,
    });
  }
};

// ============================================
// GET USER PAYMENT HISTORY (BONUS)
// ============================================
exports.getUserPayments = async (req, res) => {
  try {
    const { userId } = req.params;
    const { status, page = 1, limit = 10 } = req.query;

    if (!userId) {
      return res.status(400).json({
        success: false,
        message: "User ID is required",
      });
    }

    const query = { userId };
    if (status) {
      query.status = status;
    }

    const payments = await Payment.find(query)
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .select("-gatewayResponse -webhookData -callbackResponse"); // Don't expose sensitive data

    const count = await Payment.countDocuments(query);

    return res.status(200).json({
      success: true,
      data: payments,
      pagination: {
        total: count,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(count / limit),
      },
    });
  } catch (error) {
    console.error("Get User Payments Error:", error.message);
    return res.status(500).json({
      success: false,
      message: "Failed to fetch payments",
      error: error.message,
    });
  }
};